/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2026 i-net software GmbH, Berlin, Germany.
**/
package com.inet.taskplanner.logaction;

import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.inet.config.structure.model.LocalizedKey;
import com.inet.id.GUID;
import com.inet.logging.LogLevel;
import com.inet.taskplanner.server.api.action.ResultActionDefinition;
import com.inet.taskplanner.server.api.action.ResultActionFactory;
import com.inet.taskplanner.server.api.action.ResultActionInfo;
import com.inet.taskplanner.server.api.common.SummaryEntry;
import com.inet.taskplanner.server.api.common.SummaryInfo;
import com.inet.taskplanner.server.api.error.ValidationException;
import com.inet.taskplanner.server.api.field.Field;
import com.inet.taskplanner.server.api.field.SelectField;
import com.inet.taskplanner.server.api.field.TextField;
import com.inet.taskplanner.server.api.result.ResultFlavor;

/**
 * This log result action factory defines how the action is represented to the user and may include validation of
 * properties a user can configure.<br>
 * Creating the action instance as well as returning a summary that describes the settings is task of this
 * implementation.<br>
 * <br>
 * The log result action supports the definition of a logger name and a log level the output is printed to. Only results
 * with the flavor text are supported and will be handled.
 */
public class LogResultActionFactory extends ResultActionFactory<LogResultAction> {

    private static final String   PROPERTY_LOG_NAME  = "logname";
    private static final String   PROPERTY_LOG_LEVEL = "loglevel";

    private static final String   DEFAULT_LOG_NAME   = "Task Planner";
    private static final LogLevel DEFAULT_LOG_LEVEL  = LogLevel.STATUS; // If this is changed to something below ERROR, the visible log level must be set in configuration

    /**
     * Creates the factory and defines a unique identifier
     */
    public LogResultActionFactory() {
        super( "action.log" );
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<ResultFlavor> getSupportedFlavors( ResultActionDefinition definition ) {
        return Arrays.asList( ResultFlavor.TEXT );
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public ResultActionInfo getInformation( @Nullable GUID taskID ) {
        String name = TaskPlannerLogActionServerPlugin.MSG.getMsg( "taskplanner.logaction.name" ); // A meaningful name
        String description = TaskPlannerLogActionServerPlugin.MSG.getMsg( "taskplanner.logaction.description" ); // A short description
        URL icon = getClass().getResource( "/com/inet/taskplanner/logaction/taskplanner_logaction_32.png" ); // white and transparent icon in 32x32 pixels
        String helpkey = null; // result action does not provide an own help page
        List<Field> fields = new ArrayList<Field>(); // fields that are shown to be configured. Provide null if no settings should be done by the user

        TextField logName = new TextField( PROPERTY_LOG_NAME, TaskPlannerLogActionServerPlugin.MSG.getMsg( "taskplanner.logaction.logname" ) );
        logName.setPlaceholder( DEFAULT_LOG_NAME );
        fields.add( logName );

        ArrayList<LocalizedKey> values = new ArrayList<LocalizedKey>();
        values.add( new LocalizedKey( LogLevel.STATUS.name(), LogLevel.STATUS.name() ) );
        values.add( new LocalizedKey( LogLevel.ERROR.name(), LogLevel.ERROR.name() ) );
        values.add( new LocalizedKey( LogLevel.WARN.name(), LogLevel.WARN.name() ) );
        values.add( new LocalizedKey( LogLevel.INFO.name(), LogLevel.INFO.name() ) );
        values.add( new LocalizedKey( LogLevel.DEBUG.name(), LogLevel.DEBUG.name() ) );
        SelectField logLevel = new SelectField( PROPERTY_LOG_LEVEL, TaskPlannerLogActionServerPlugin.MSG.getMsg( "taskplanner.logaction.loglevel" ), values );
        logLevel.setValue( DEFAULT_LOG_LEVEL.name() );
        fields.add( logLevel );

        return new ResultActionInfo( getExtensionName(), name, description, icon, helpkey, fields );
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void validate( @Nonnull ResultActionDefinition definition, @Nullable GUID taskID ) throws ValidationException {
        // Nothing to validate
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected LogResultAction createInstanceFrom( @Nonnull ResultActionDefinition definition, @Nullable GUID taskID ) {
        String logName = getResultingLogName( definition );
        LogLevel logLevel = getResultingLogLevel( definition );
        return new LogResultAction( logName, logLevel );
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public SummaryInfo getSummary( @Nonnull ResultActionDefinition definition ) {
        List<SummaryEntry> entries = new ArrayList<SummaryEntry>();

        // Create summary with no extra label but with a short description.
        String description = TaskPlannerLogActionServerPlugin.MSG.getMsg( "taskplanner.logaction.summary", getResultingLogName( definition ), getResultingLogLevel( definition ).name() );
        entries.add( new SummaryEntry( null, description ) );

        return new SummaryInfo( entries );
    }

    /**
     * Determines the resulting name of the logger. Will use the fallback if no log name is set.
     * @param definition the definition of the result action to get the properties from.
     * @return the resulting log name
     */
    private String getResultingLogName( ResultActionDefinition definition ) {
        String logName = definition.getProperty( PROPERTY_LOG_NAME );
        if( logName == null || logName.trim().isEmpty() ) {
            logName = DEFAULT_LOG_NAME;
        }
        return logName;
    }

    /**
     * Determines the resulting log level. Will use the fallback if the save log level name is not valid.
     * @param definition the definition of the result action to get the properties from.
     * @return the resulting log level
     */
    private LogLevel getResultingLogLevel( ResultActionDefinition definition ) {
        LogLevel logLevel = DEFAULT_LOG_LEVEL;
        try {
            String logLevelName = definition.getProperty( PROPERTY_LOG_LEVEL );
            logLevel = LogLevel.valueOf( logLevelName );
        } catch( IllegalArgumentException iae ) {
            // Saved log level is not valid
            TaskPlannerLogActionServerPlugin.LOGGER.error( iae );
        }
        return logLevel;
    }
}
