/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2025 i-net software GmbH, Berlin, Germany.
**/
package com.inet.authentication.oauth.ex;

import java.io.IOException;
import java.net.URL;
import java.net.URLConnection;

import javax.annotation.Nonnull;

import com.inet.authentication.AuthenticationDescription;
import com.inet.authentication.oauth2.api.OAuthServerDescription;

/**
 * The description of the OAuth service. This sample is very simple with mostly default values for a fantasy service supergit.
 * This will not work with every service. Some service required more settings. 
 * See in the documentation of the parent class for more settings or contact the support.
 */
public class OAuthExServerDescription implements OAuthServerDescription {

    // a unique name for the authentication service. It is to differ between 2 login IDs from different sources.
    // "adfs", "facebook", "github", "google" and "system" are already in use.
    static final String NAME                     = "supergit";

    // a unique key inside all OAuth settings
    static final String OAUTH_AUTHENTICATION_URL = NAME + ".authentication.server";

    // a unique key inside all OAuth settings
    static final String OAUTH_TOKEN_URL          = NAME + ".token.server";

    // a unique key inside all OAuth settings
    static final String OAUTH_USERINFO_URL          = NAME + ".userinfo.server";

    /**
     * {@inheritDoc}
     */
    @Override
    public @Nonnull String name() {
        return NAME;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public @Nonnull String getAuthenticationURL( @Nonnull AuthenticationDescription config ) {
        // Get the URL to the oauth service from the configuration like "https://oauth.supergit.com/login/oauth/authorize"
        // Can be a hard coded URL if you does not want to customize it through the UI.
        return config.getSettings().get( OAUTH_AUTHENTICATION_URL );
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getTokenURL( @Nonnull AuthenticationDescription config ) {
        // Get the URL to the token service from the configuration like "https://oauth.supergit.com/login/oauth/token"
        // Can be a hard coded URL if you does not want to customize it through the UI.
        return config.getSettings().get( OAUTH_TOKEN_URL );
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public URLConnection getDataConnection( @Nonnull AuthenticationDescription config, String accessToken ) throws IOException {
        // Get the URL to the token service from the configuration like "https://oauth.supergit.com/login/oauth/token"
        // Can be a hard coded URL if you does not want to customize it through the UI.
        URL url = new URL( config.getSettings().get( OAUTH_USERINFO_URL ) );
        URLConnection conn = url.openConnection();
        conn.setRequestProperty( "Authorization", "Bearer " + accessToken );
        return conn;
    }
}
