/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2025 i-net software GmbH, Berlin, Germany.
**/
package com.inet.automaticconfigrestore;

import java.util.Properties;
import java.util.prefs.BackingStoreException;

import javax.annotation.Nullable;

import com.inet.classloader.I18nMessages;
import com.inet.config.Configuration;
import com.inet.config.ConfigurationManager;
import com.inet.logging.LogManager;
import com.inet.logging.SystemEventLog;
import com.inet.plugin.PluginInfo;
import com.inet.plugin.ServerPlugin;
import com.inet.plugin.ServerPluginManager;
import com.inet.plugin.ServerPluginManagerListener;
import com.inet.plugin.veto.VetoType;
import com.inet.process.ProcessStarter;

/**
 * Server plugin for an automatic backup and restore of the configuration using the persistence.<br>
 */
@PluginInfo( //
id = "configuration.automaticconfigrestore", //
dependencies = "", //
optionalDependencies = "", //
packages = "", //
group = "samples", // 
version = "25.10.198", //
icon = "com/inet/maintenance/automaticconfigrestore/auto_config_restore_48.png", //
flags = "" //
)
public class AutomaticConfigRestoreServerPlugin implements ServerPlugin {

    public static final I18nMessages MSG = new I18nMessages( "com.inet.automaticconfigrestore.structure.i18n.ConfigStructure", AutomaticConfigRestoreServerPlugin.class );

    public AutomaticConfigRestoreServerPlugin() {
        super();

        Properties properties = ConfigStoreChangeManager.getStoredConfiguration();
        if ( properties == null ) {
            return; // nothing to do
        }

        Configuration configuration = ConfigurationManager.getInstance().getCurrent();
        configuration.putAll( properties );

        try {
            configuration.flush();

            final int exitCode = Integer.valueOf( System.getProperty( ProcessStarter.RESTART_EXITCODE ) ).intValue();
            SystemEventLog.ServerRestarted.log();
            LogManager.getConfigLogger().status( "Restarting server due to automatic configuration restorement!" );
            System.exit( exitCode );

        } catch( BackingStoreException e ) {
            // There was a much greater problem
            LogManager.getConfigLogger().debug( e );
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void registerExtension( ServerPluginManager spm ) {

        spm.register( ServerPluginManagerListener.class, new ServerPluginManagerListener() {
            @Override
            public void vetoFinished( @Nullable VetoType type ) {
                if( type == null ) { // if all vetos are finished
                    // Start checking for config changes after the user manager was inited.
                    ConfigurationManager.getInstance().addConfigurationChangeListener( new ConfigStoreChangeManager() );
                }
            }
        } );
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void init( ServerPluginManager spm ) {
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void reset() {
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void restart() {
    }
}
