/*
  i-net software provides programming examples for illustration only, without warranty
  either expressed or implied, including, but not limited to, the implied warranties
  of merchantability and/or fitness for a particular purpose. This programming example
  assumes that you are familiar with the programming language being demonstrated and
  the tools used to create and debug procedures. i-net software support professionals
  can help explain the functionality of a particular procedure, but they will not modify
  these examples to provide added functionality or construct procedures to meet your
  specific needs.
  © i-net software 1998-2011
*/

package com.inet.samples.session.datasource;

import java.io.IOException;
import java.io.Serializable;
import java.security.Principal;

import com.inet.report.config.datasource.DataSourceConfiguration;
import com.inet.report.config.datasource.DataSourceConfigurationManager;

import jakarta.servlet.Filter;
import jakarta.servlet.FilterChain;
import jakarta.servlet.FilterConfig;
import jakarta.servlet.ServletException;
import jakarta.servlet.ServletRequest;
import jakarta.servlet.ServletResponse;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletRequestWrapper;

/**
 * An implementation of jakarta.servlet.Filter that accept injected user data to create a session scope datasource. This
 * can be used if the users are already login in a frontend server and you want use the same account in the backend.
 */
public class UserInjection implements Filter {

    // Do you want another name for the datasource? Set it here:
    private static final String DS_NAME = "datasource";

    /**
     * {@inheritDoc}
     */
    @Override
    public void init( FilterConfig config ) throws ServletException {
        // nothing
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void doFilter( ServletRequest request, ServletResponse response, FilterChain chain ) throws IOException, ServletException {
        final HttpServletRequest req = (HttpServletRequest)request;

        // read the Properties from the request 
        String user = request.getParameter( "user" );
        if( !isEmpty( user ) ) {
            String password = request.getParameter( "password" );
            String host = request.getParameter( "host" );
            String port = request.getParameter( "port" );
            String database = request.getParameter( "database" );

            // Create the datasource if all needed was set
            if( !DataSourceConfigurationManager.exists( DS_NAME, DataSourceConfigurationManager.USER_SESSION ) && !isEmpty( password )
                && !isEmpty( host ) && !isEmpty( port ) && !isEmpty( database ) ) {
                DataSourceConfiguration dsc;
                dsc = DataSourceConfigurationManager.createDataSourceConfiguration( DS_NAME, DataSourceConfigurationManager.USER_SESSION );
                dsc.setUser( user );
                dsc.setDriverClassname( "com.mysql.cj.jdbc.Driver" );
                dsc.setUrl( "jdbc:mysql://{host}:{port}/{database}" ); // URL for MySQL with place holder
                dsc.setPassword( password );
                dsc.addProperty( "host", host );
                dsc.addProperty( "port", port );
                dsc.addProperty( "database", database );
                dsc.save();
            }
            // save the Principal in the session for future requests which do not have the injected data
            req.getSession( true ).setAttribute( UserInjection.this.getClass().getName(), new InjectedPrincipal( user ) );
        }

        // create a wrapper to pass the Principal because there is no setter
        request = new HttpServletRequestWrapper( req ) {
            @Override
            public Principal getUserPrincipal() {
                return (Principal)req.getSession( true ).getAttribute( UserInjection.this.getClass().getName() );
            }
        };

        // call the i-net Clear Reports servlet
        chain.doFilter( request, response );
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void destroy() {
        // nothing
    }

    /**
     * Check if the string is null or empty
     * @param value the string
     * @return true, if null or empty
     */
    private static boolean isEmpty( String value ) {
        return value == null || value.isEmpty();
    }

    /**
     * Simple implementation of Principal
     */
    static class InjectedPrincipal implements Principal, Serializable {

        private final String user;

        /**
         * Create Principal with a user name
         * @param user the login user name
         */
        InjectedPrincipal( String user ) {
            this.user = user;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String getName() {
            return user;
        }

    }
}
