import {Kytos} from "./kytos.js";
import {KytosElement} from './kytos-element.js';
import {KytosAttribute} from "./kytos-attribute.js";

/**
 * Mixin for k-repeat element and k-repeat-item attribute. Contains the base implementations for syncing the list values with the template nodes in the UI
 */
let KRepeatMixin = (superclass) => class extends superclass {
	
	/** 
	 * COPY!!! of the list containing the values that should be repeated in the DOM
	 */
	_itemListModel = undefined;
	
	/**
	 * The list with COPY!!! of nodes that should be repeated in the DOM. 
	 */
	_templateNodes = [];
	
	/**
	 * Comment node where the repeated items should be added after in the DOM
	 */
	_templateMarkerStart = undefined;

	/**
	 * Comment node where the repeated items should be added before in the DOM
	 */
	_templateMarkerEnd = undefined;
	
	/**
	 * Reference to the Kytos Element instance. Will be set by the k-repeat-item attribute
	 */
	_kytosElement = this;
	
	/**
	 * internal item cache to preserve nodes that are used for a value from the _itemListModel 
	 */
	#itemCache = new Map();
		
	/**
	 * Sets the item list. Must be of type 'Array'. A copy will be created and the list updates the UI. 
	 * @param itemList the new item list with value to be repeated
	 */
	set itemList(itemList) {
		if (!Array.isArray(itemList)) {
			throw new Error('An array must be given as item list!');
		}
		let previousItemList = this._itemListModel;
		this._itemListModel = [...itemList];
		
		this._syncItemListChanges(previousItemList, this._itemListModel);
	};

	/**
	 * Returns a copy of the current known list of items
	 * @return a copy of the current known list of items
	 */
	get itemList() {
		if (this._itemListModel == undefined) {
			return undefined;
		}
		return [...this._itemListModel];
	};
	
	/**
	 * Syncs the list with the DOM nodes.
	 * It will first remove all entries that are no longer available.
	 * Afterwards, all items in newList will be put in the correct order and the value binding is updated.
	 * Moving a value in the _itemListModel will result in updating all item bindings.
	 * @param oldList the old list of values
	 * @param newList the new list of values
	 */
	_syncItemListChanges(oldList = [], newList = []) {
		for (let oldItem of oldList) {
			if (!newList.includes(oldItem)) {
				let oldItemEntries = this.#itemCache.get(oldItem);
				if (oldItemEntries != undefined) {
					for (let node of oldItemEntries) {
						this.#removeAttributeBinding(node);
						node.remove();
					}
					this.#itemCache.delete(oldItem);
				}
			}
		}
		
		let tnc = this._templateNodes.length;
		
		for (let i=0; i<newList.length; i++) {
			let newItem = newList[i];
			let newItemEntries = this.#itemCache.get(newItem);
			if (newItemEntries == undefined) {
				newItemEntries = [];
				for (let node of this._templateNodes) {
					let clonedNode = node.cloneNode(true);
					clonedNode.kRepeatNode = true;
					const treeWalker = document.createTreeWalker(clonedNode, NodeFilter.SHOW_ELEMENT);
			        while ( treeWalker.nextNode() ) {
			            const subNode = treeWalker.currentNode;
						subNode.kRepeatNode = true;
			        }
					newItemEntries.push(clonedNode);
				}
				this.#itemCache.set(newItem, newItemEntries);
			}
			
			let existingNode = undefined;
			let sibling = this._templateMarkerStart;
			for (let s = 0; s < i; s ++) {
				for (let c = 0; c < tnc; c ++) {
					sibling = sibling.nextSibling;
				}				
			}
			existingNode = (sibling||{}).nextSibling;
			
			for (let node of newItemEntries) {
				this.#replaceItemBindings(node, newItem, i);
			}
			if (newItemEntries[0] == existingNode) {
				continue; // same node at same index;
			}
			
			for (let node of newItemEntries) {
				node.remove();
				this._templateMarkerEnd.parentNode.insertBefore(node, existingNode || this._templateMarkerEnd);
			}
			
		}
		
	};

	/**
	 * Replaces all mustach bindings in attributes of the given node. Supports {{index}}, {{item}} and {{item.abc.def}} values.
	 * @param node the node to check the attributes at
	 * @param item the item value
	 * @param index the index of the value in the _itemListModel
	 */
	#replaceAttributeNodeBinding(node, item, index) {
		if (node.nodeType != Node.ELEMENT_NODE) {
			return;
		}
		let attributes = node.attributes;
		for (let attr of attributes) {
			let name = attr.nodeName;
			let value = attr.nodeValue;
			
			if (node.kRepeatAttributeOriginals == undefined) {
				node.kRepeatAttributeOriginals = {};
			} 
			
			let originalContent = node.kRepeatAttributeOriginals[name];
			if (originalContent != undefined) {
				value = originalContent;
			}
			
			let startIndex = value.indexOf('{{');
			while (startIndex >= 0) {
			    let endIndex = value.indexOf('}}', startIndex);
			    if (endIndex >= 0) {
					let originalContent = node.kRepeatAttributeOriginals[name];
					if (originalContent == undefined) {
						node.kRepeatAttributeOriginals[name] = value;
					}
					
			        let property = value.substring(startIndex+2, endIndex);
					let replaceText = undefined;
					
					if (property == 'index') {
						replaceText = index;
					} else if (property == 'item') {
						replaceText = item;
					} else if (property.startsWith('item.')) {
						replaceText = this.#getItemValue(property.substring(5), item);
					}
					let replaceData  = value.substring(0,startIndex);
					replaceData += replaceText;
					replaceData += value.substring(endIndex+2);
					value = replaceData;
			    }
			    startIndex = value.indexOf('{{', startIndex+2);
			}
			if (attr.nodeValue != value) {
				attr.nodeValue = value;
			}
		}
	};

	/**
	 * Replaces all mustach bindings in text content of the given node. Supports {{index}}, {{item}} and {{item.abc.def}} values.
	 * @param node the node to check the content at
	 * @param item the item value
	 * @param index the index of the value in the _itemListModel
	 */
	#replaceTextNodeBinding(node, item, index) {
		if (node.nodeType != Node.TEXT_NODE) {
			return;
		}
		let originalContent = node.parentNode.kOriginalContentRepeat;
		if (originalContent == undefined) {
			originalContent = node.data.toString();
			node.parentNode.kOriginalContentRepeat = originalContent;
		}
		let nodeData = originalContent;
		let startIndex = nodeData.indexOf('{{');
		while (startIndex >= 0) {
		    let endIndex = nodeData.indexOf('}}', startIndex);
		    if (endIndex >= 0) {
		        let property = nodeData.substring(startIndex+2, endIndex);
				let replaceText = undefined;
				
				if (property == 'index') {
					replaceText = index;
				} else if (property == 'item') {
					replaceText = item;
				} else if (property.startsWith('item.')) {
					replaceText = this.#getItemValue(property.substring(5), item);
				}
				let replaceData  = nodeData.substring(0,startIndex);
				replaceData += replaceText;
				replaceData += nodeData.substring(endIndex+2);
				nodeData = replaceData;
		    }
		    startIndex = nodeData.indexOf('{{', startIndex+2);
		}
		if (node.data != nodeData) {
			node.data = nodeData;
		}
	};

	/**
	 * Replaces all mustach bindings in text content and attributes of the given node and subnodes. Supports {{index}}, {{item}} and {{item.abc.def}} values.
	 * @param node the itemNode to recursivly replace attribute and text content bindings
	 * @param item the item value
	 * @param index the index of the value in the _itemListModel
	 */
	#replaceItemBindings(itemNode, item, index) {
		this.#replaceTextNodeBinding(itemNode, item, index);
		this.#replaceAttributeNodeBinding(itemNode, item, index);
		const treeWalker = document.createTreeWalker(itemNode, NodeFilter.SHOW_TEXT | NodeFilter.SHOW_ELEMENT, (node) => {
			if (node.tagName == 'K-REPEAT') {
				return NodeFilter.FILTER_REJECT;
			}
			if (node.nodeType == Node.ELEMENT_NODE) {
				if (!node.kRepeatNode) {
					return NodeFilter.FILTER_REJECT;
				}
			}
			return NodeFilter.FILTER_ACCEPT;
		});
		
		let kytosAttributeNodes = new Map();

	    while ( treeWalker.nextNode() ) {
	        const node = treeWalker.currentNode;
			this.#replaceAttributeNodeBinding(node, item, index);
			this.#replaceTextNodeBinding(node, item, index);
			
			if (node.nodeType == Node.ELEMENT_NODE) {
				for (let attrName of node.getAttributeNames()) {
		            if (KytosAttribute._hasAttributeDefined(attrName)) {
						if (!kytosAttributeNodes.has(attrName)) {
							kytosAttributeNodes.set(attrName, []);
						}
						kytosAttributeNodes.get(attrName).push(node);
		            }
		        }
			}
	    }
		
		for (let [attrName, nodes] of kytosAttributeNodes) {
			for (let node of nodes) {
				KytosAttribute._unbindAttribute(attrName, this._kytosElement, node);
				KytosAttribute._bindAttribute(attrName, this._kytosElement, node);
			}
		}
	};
	
	#removeAttributeBinding(itemNode) {
		const treeWalker = document.createTreeWalker(itemNode, NodeFilter.SHOW_TEXT | NodeFilter.SHOW_ELEMENT, (node) => {
			if (node.tagName == 'K-REPEAT') {
				return NodeFilter.FILTER_REJECT;
			}
			if (node.nodeType == Node.ELEMENT_NODE) {
				if (!node.kRepeatNode) {
					return NodeFilter.FILTER_REJECT;
				}
			}
			return NodeFilter.FILTER_ACCEPT;
		});
		
		let kytosAttributeNodes = new Map();

	    while ( treeWalker.nextNode() ) {
	        const node = treeWalker.currentNode;
			if (node.nodeType == Node.ELEMENT_NODE) {
				for (let attrName of node.getAttributeNames()) {
		            if (KytosAttribute._hasAttributeDefined(attrName)) {
						if (!kytosAttributeNodes.has(attrName)) {
							kytosAttributeNodes.set(attrName, []);
						}
						kytosAttributeNodes.get(attrName).push(node);
		            }
		        }
			}
	    }
		
		for (let [attrName, nodes] of kytosAttributeNodes) {
			for (let node of nodes) {
				KytosAttribute._unbindAttribute(attrName, this._kytosElement, node);
			}
		}
	}

	/**
	 * Returns the value of the given property name in the provided model. Splits the property name at dots to access inner values.
	 * @param propName the name of the property
	 * @param model the model to get the value from
	 * @return the value in the mode or undefined if not found
	 */
	#getItemValue(propName, model) {
		let index = propName.indexOf('.');
	    let subProps = undefined; 
	    if (index>0) { 
	        subProps = propName.substring(index+1);
	        propName = propName.substring(0,index);
	    }
	    let value = model[propName];
	    if (subProps != undefined) {
	        return this.#getItemValue(subProps, value);
	    }
	    return value;
	};	
};

/**
 * k-repeat element, build upon KytosElement.
 * 
 * Supports bindings for {{index}}, {{item}} and {{item.abc.def}}.
 * The given array can either contain simple datatypes like strings or objects containing simple types or other object.
 * 
 * Usage:
 * <k-repeat>
 *     <div>fixed content</div>
 *     <div k-repeat>repeated line 1 at index {{index}}}</div>
 *     <div k-repeat>repeated line 2 at index {{index}}}</div>
 *     <div k-repeat data-key="{{item.key}}">{{item.label}}</div>
 *     <div>fixed content</div>
 * </k-repeat>
 * 
 * this.querySelector('k-repeat').itemList = myArray;
 * 
 * Limitations:
 * - Only usable as tag element
 * - itemList must be set via javascript
 * - changes must be notified by settings the itemList again
 */
export class KytosRepeat extends KRepeatMixin(KytosElement) {
	
	/**
	 * Called when the element is attached to the DOM. It will collect its inner elements as template nodes and creates comment marker as placeholders for the items.
	 */
	attached() {
		let firstKRepeatNodeFound = false;

		let repeatChildren = [];

		const treeWalker = document.createTreeWalker(this, NodeFilter.SHOW_ELEMENT, (node) => {
			if (node.tagName == 'K-REPEAT') {
				return NodeFilter.FILTER_REJECT;
			}
			if (node.parentNode != undefined && node.parentNode.hasAttribute('k-repeat')) {
				return NodeFilter.FILTER_REJECT;
			}
			return NodeFilter.FILTER_ACCEPT;
		});
		while ( treeWalker.nextNode() ) {
		    const subNode = treeWalker.currentNode;
			
			if (subNode.hasAttribute('k-repeat')) {
				if (!firstKRepeatNodeFound) {
					
					if (this._templateNodes.length>0) {
						throw new Error('All repeatable nodes (tagged with the "k-repeat" attributes) must be in a single block. It ist not allowed to split it with non repeatable nodes.');
					}
					
					firstKRepeatNodeFound = true;

					this._templateMarkerStart = document.createComment('k-repeat-start');
					subNode.before(this._templateMarkerStart);
					this._templateMarkerEnd = document.createComment('k-repeat-end');
					subNode.after(this._templateMarkerEnd);
					let node = subNode.cloneNode(true);
					this._templateNodes.push(node);
					repeatChildren.push(subNode);
				} else {
					this._templateMarkerEnd.remove();
					subNode.after(this._templateMarkerEnd);
					let node = subNode.cloneNode(true);
					this._templateNodes.push(node);
					repeatChildren.push(subNode);
				}
			} else {
				if (firstKRepeatNodeFound) {
					firstKRepeatNodeFound = false;
				}
			}
		}

		if (repeatChildren.length==0) {
			throw new Error('No repeatable element found (tagged with "k-repeat" attribute)');
		}
		for (let child of repeatChildren) {
			child.remove();
		}
	};
	
};

if ( !customElements.get('k-repeat') ) {
    customElements.define('k-repeat', KytosRepeat);    
}

/**
 * k-repeat-item attribute, build upon KytosAttribute.
 * 
 * Value of this attribute is the property name from the parent KytosElement instance. If an array from dataBindingModel is provided, it is automatically watched for changes.
 * Supports bindings for {{index}}, {{item}} and {{item.abc.def}}.
 * The given array can either contain simple datatypes like strings or objects containing simple types or other object.
 * 
 * Usage:
 * <div k-repeat-item="dataBindingModel.myArray">
 *     <div>repeated line at index {{index}}} for {{}}</div>
 * </div>
 * 
 * Limitations:
 * - Only usable as attribute
 * - Only one node can be repeated. A subtree is possible.
 * - Value must be specified an an array of the surrounding KytosElement
 * - Automatic UI updates are only possible if the array is part of the dataBindingModel of the parent KytosElement
 * - Whether the node of this attribute will be detached from the DOM, ist observed via the surrounding KytosElement. 
 */
export class KytosRepeatItem extends KRepeatMixin(KytosAttribute) {

	/**
	 * Returns that the inner content are templates an should not be bound
	 * @return true.
	 */
	static skipInnerNodes() {
		return true;
	}
	
	/**
	 * Called when the node with this attribute is attached to the DOM. It will collect its node as template nodes and creates comment marker as plceholders for the items.
	 * Aftwerwards a MutationObserver and a watcher on the dataBindingModel of the surrounding KytosElement will be registered to automatically update the UI when the array changes
	 */	
	attached(kytosElement, node, attrName) {
		let bindingProperty = node.getAttribute(attrName);
		if (bindingProperty == undefined || bindingProperty.length==0) {
			throw new Error('An attribute value must be defined, containing a property (Array) in in the dataBindingModel of the parent KytosElement.');
		}
		
		this._kytosElement = kytosElement;
		
		this._templateMarkerStart = document.createComment(attrName + 'start');
		this._templateMarkerEnd = document.createComment(attrName + 'end');
		
		node.before(this._templateMarkerStart);
		node.after(this._templateMarkerEnd);
		
		this._templateNodes.push(node.cloneNode(true));
		node.remove();

		this.itemList = Kytos.getModelValue(bindingProperty, kytosElement);
		
		let deregisterWatcher = undefined;
		if (bindingProperty.startsWith('dataBindingModel.')) {
			deregisterWatcher = kytosElement.watch(bindingProperty.substring('dataBindingModel.'.length), (newValue) => {
				this.itemList = newValue;
			});
		}

		let observer = Kytos.onNodeRemoved(this._templateMarkerStart, (node) => {
			this._templateMarkerStart.remove();
			this._templateMarkerEnd.remove();
			observer.disconnect();
			if (deregisterWatcher != undefined) {
				deregisterWatcher();
			}
		}, kytosElement);
	};
	
};

KytosAttribute.define('k-repeat-item', KytosRepeatItem, false);

