import { Kytos } from './kytos.js';
import { EventEmitter } from './eventemitter.js';

/**
 * Data Service instance that can be used via DataService.execute(options) call.
 * 
 * It handles the JSON encryption, decryption and utf8 converting and provides a convenient API.
 * 
 * Usage:
 *   DataService.execute(options);
 * 
 * Usage with await in async function:
 * async myFunc() {
 * 	 await DataService.execute(options);
 *   console.log('request executed and handled');
 * };
 * 
 * Options:
 * - command: an command that is backed by a ServiceMethod. Default: undefined
 * - data: data obejct to be send. Default: {}
 * - method: HTTP method. Default: 'POST'
 * - url: base URL to be called. Default: '?method=',
 * - showWaitingOverlay: Whether to show the waiting overlay during loading. Default: true,
 * - success: Function with the response data in success case. Default: (responseData) => {},
 * - error: Function with the error messsage in error case. Default: (errorMessage) => {},
 * - beforeExecute: Function, called before the requests start. Default: () => {},
 * - afterExecute: Function, called after the requests finished. Default: () => {},
 * - beforeError: Function, called before an error is reported. Returning true will suppress the error. Default: (errorMessage) => {},
 * - afterError: Function, called after an error is reported. Default: (errorMessage) => {},
 * - headers: Map of headers. Default: {'Content-Type': 'application/json'},
 * - timeout: Timeout in milli seconds. Default: 60000
 */
export class DataService {

	/*****************************************************************************************************************************************************************************************************************************/
	/*****************************************************************************************************************************************************************************************************************************/
	/** API methods and members                                                                                                                                                                                                 **/
	/*****************************************************************************************************************************************************************************************************************************/
	/*****************************************************************************************************************************************************************************************************************************/

	/**
	 * Event that is fired when a request started
	 */
	static EVENT_REQUEST_STARTED = 'dataservice.request.started';
	/**
	 * Event that is fired when a request has ended
	 */
	static EVENT_REQUEST_ENDED = 'dataservice.request.ended';

	/**
	 * Executes a request. Providing options will define the target ServiceMethod and data to be send and retrieved.
	 * @param opts the options of the request
	 */
	static async execute(opts = {}) {
		if (opts.command == undefined && opts.url == undefined) {
			throw new Error('A command or url must be specified for the request to be executed.');
		}

		const defaults = {
			data: {},
			command: undefined,
			method: 'POST',
			url: '?method=',
			showWaitingOverlay: true,
			success: (responseData) => { },
			error: (errorMessage) => { },
			beforeExecute: () => { },
			afterExecute: () => { },
			beforeError: (errorMessage) => { },
			afterError: (errorMessage) => { },
			headers: {
				'Content-Type': 'application/json'
			},
			timeout: 60000
		};

		const options = { ...defaults, ...opts };

		try {
			EventEmitter.fire(DataService.EVENT_REQUEST_STARTED, {
				showWaitingOverlay: options.showWaitingOverlay
			});
			options.beforeExecute();

			// Increase timeout when uploading attachments
			if (options.attachments != undefined && options.attachments.length > 0) {
				var contentsize = 0;
				for (var k = 0; k < options.attachments.length; k++) {
					var attach = options.attachments[k];
					if (attach.file) {
						contentsize = contentsize + attach.file.size;
					}
				}
				// Double ISDN has a max 16384 Bytes per second transfer rate. 30 seconds tolerance. @see JsonRpcClient
				var timeoutSec = Math.round(contentsize / 16384) + 30; // timeout in seconds
				timeoutSec = Math.min(timeoutSec, 1800); // Maximal 30 min Timeout
				options.timeout = Math.max(60000, timeoutSec * 1000);
			}

			if (options.attachments != undefined && options.attachments.length > 0) {
				delete options.headers; // For multipart/formdata, no headers are allowed to be provied, otherwise the boundaries are missing

				var formData = new FormData();
				var largeContent = [];
				for (let i=0; i<options.attachments.length; i++) {
					let att = options.attachments[i];
					formData.append('attachment' + i, att.file, att.name);
					largeContent[i] = {
						name: att.name,
						lastModified: att.file.lastModifiedDate == undefined ? (att.file.lastModified || new Date().getTime()) : att.file.lastModifiedDate.getTime(),
						attachmentType: att.attachmentType
					};
					var varNames = Object.keys(att);
					for (var v = 0; v < varNames.length; v++) {
						var varName = varNames[v];
						if (varName.indexOf('$_') == 0) {
							largeContent[i][varName.substring(2)] = att[varName];
						}
					}
				}

				options.data.attachments = largeContent;
				formData.append('json', new Blob([DataService.#convertData(JSON.stringify(options.data))], { "type": "application/binary-json" }));
				options.data = formData;
			} else {
				options.data = DataService.#convertData(JSON.stringify(options.data));
			}

			// Create AbortController for timeout
			const controller = new AbortController();
			const timeoutId = setTimeout(() => controller.abort(getMsg("timeout")), options.timeout);

			// Execute request
			const response = await fetch(options.url + (options.command || ''), {
				method: options.method,
				headers: options.headers,
				signal: controller.signal,
				body: options.data
			});
			clearTimeout(timeoutId);

			let data = await response.bytes();
			data = new Uint8Array(data).buffer;
			data = DataService.#extractData(data, response.headers);

			if (response.ok) {
				await options.success(data);
			} else {
				DataService.#handleError(options, data, response.headers, response.status);
			}
		} catch (exception) {
			console.error(exception);
			DataService.#handleError(options, getMsg("noconnection"), undefined, -1);
		} finally {
			options.afterExecute();
			EventEmitter.fire(DataService.EVENT_REQUEST_ENDED, {
				showWaitingOverlay: options.showWaitingOverlay
			});
		}
	};

	/*****************************************************************************************************************************************************************************************************************************/
	/*****************************************************************************************************************************************************************************************************************************/
	/** API END                                                                                                                                                                                                                 **/
	/*****************************************************************************************************************************************************************************************************************************/
	/*****************************************************************************************************************************************************************************************************************************/

	/**
	 * Converts the input data end encrypts them
	 * @param data the input data
	 * @returns the encrypted data as utf8 array
	 */
	static #convertData(data) {
		var enc = Kytos.encrypt(data);
		var utf8Bytes = Kytos.toUTF8Array(enc);
		return new Uint8Array(utf8Bytes);
	};

	/**
	 * Extracts the response data, according to the content type and data type
	 * @param data the data of the response
	 * @param headers the headers map
	 * @return the extracted JSON data object
	 */
	static #extractData(data, headers) {
		if (data) {
			if (headers.get('content-type') == 'application/binary-json' || headers.get('content-type') == 'application/error-json') {
				if (data instanceof ArrayBuffer) {
					data = Kytos.extractJsonBytes(data);
				} else {
					data = Kytos.extractJson(data);
				}
			} else {
				if (data instanceof ArrayBuffer) {
					data = Kytos.utf8ArrayToStr(new Uint8Array(data));
				}
				if (headers.get('content-type') != undefined && headers.get('content-type').indexOf('text/html') == 0) {
					data = Kytos.html2text(data);
				}
			}
		}
		return data;
	};

	/**
	 * Extracts the error message, according to the content type and data type
	 * @param data the data of the response/error
	 * @param headers the headers map
	 * @param status the response status
	 * @return the extracted error message
	 */
	static #extractErrorMessage = function (data, headers, status) {
		if (data && data.cmd == 'error') {
			data = JSON.parse(data.json);
		} else {
			if (status == undefined) {
				var msgNoConnection = getMsg("noconnection");
				if (data != undefined) {
					data = msgNoConnection + " " + data;
				} else {
					data = msgNoConnection;
				}
			} else if (status === 0 || (status == -1 && data == undefined)) {
				var msgTimeout = getMsg("timeout");
				if (data != undefined) {
					data = msgTimeout + " " + data;
				} else {
					data = msgTimeout;
				}
			} else {
				var msg = getMsg("error");
				if (status == undefined) {
					status = "";
				}
				if (status == -1 && data != undefined && typeof data === "object" && "message" in data) {
					data = data.message;
				}
				if (data == undefined || !data.length) {
					if (status == 401) {
						data = getMsg("unauthorized");
					} else {
						data = getMsg("unknown");
					}
				}
				if (data != undefined) {
					data = msg + " " + data;
				} else {
					data = status + " " + msg;
				}
			}
		}
		return data;
	};

	/**
	 * Handles an error by extracting the error message and checking the beforeError callback
	 * @param options the options of the request
	 * @param data the data of the response/error
	 * @param headers the headers map
	 * @param status the response status
	 */
	static #handleError(options, data, headers, status) {
		let errorMessage = DataService.#extractErrorMessage(data, headers, status);
		const shouldSuppressError = options.beforeError(errorMessage);
		if (!shouldSuppressError) {
			options.error(errorMessage);
		}
		options.afterError(errorMessage);
	};

};
