import {Kytos} from "./kytos.js";
import {KytosAttribute} from "./kytos-attribute.js";

/**
 * Base implementation of a custom HTML WebComponent with convenient Kytos API
 * 
 * <h2>Usage</h2>
 * <h3>WebComponent</h3>
 * <p>Create a script with the name my-test.js:</p>
 * <pre>
 * import {KytosElement} from './kytos-element.js';
 * 
 * export class MyTest extends KytosElement {
 * 
 *     template = `<p>{{dataBindingModel.myValue}}</p>`; // Template in back ticks to allow better formatting with line breaks and so on
 *     
 *     dataBindingModel = { 
 *         myValue: 'Hello world'
 *     };
 *     
 *     init() {
 *         // Initialize what is required
 *         this.dataBindingModel.myValue += ' and the rest of the universe!';
 *     };
 * 
 * }
 * 
 * customElements.define('my-test', MyTest);
 * </pre>
 * 
 * <h3>Referencing</h3>
 * <p>In your initial HTML file:</p>
 * <pre>
 * <script src="weblib/my-test.js" type="module"></script>
 * <my-test></my-test> 
 * </pre>
 * <p>Or in another component:</p>
 * <pre>
 * import {MyTest} from './my-test.js';
 * ...
 * template = `<my-test></my-test>`;
 * ... 
 * </pre>
 */
export class KytosElement extends HTMLElement {

    
    /*****************************************************************************************************************************************************************************************************************************/
    /*****************************************************************************************************************************************************************************************************************************/
    /** API methods and members                                                                                                                                                                                                 **/
    /*****************************************************************************************************************************************************************************************************************************/
    /*****************************************************************************************************************************************************************************************************************************/
    
    /**
     * An HTML template. Use back ticks for better formatting.
     */
    template = undefined;
    
    /**
     * A default model that is automatically observed for changes. The values may be primitives or objects of primitives. (e.g. no Date instance possible)
     * These values can be used in the template with one-way-data-bindung in mustach syntax (e.g. <code>{{myValue}}</code>) or with k-bind (e.g. <code><span k-bind="myValue"></span><input type="text" k-bind="myValue"/></code>)
     */
    dataBindingModel = {};

    /**
     * Static function to return a fixed list of attributes that are observed for the current HTML tag. Changes to these attributes will call the attributeChangedCallback function.
     * @returns {string[]} a list of attribute names that will be observed
     */
    static get observedAttributes() {
        return [];
    };
    
    /**
     * Called when an attribute, previously registered in observedAttributes(), was changes. This function is called initially for each attribute where oldValue and newValue are the same.
     * @param {string} attributeName the name of the changed attribute
     * @param {string|null} oldValue the previous value
     * @param {string|null} newValue the new value
     */
    attributeChangedCallback(attributeName, oldValue, newValue) {};
    
    /**
     * Called the first time, this WebComponent is connected to the DOM. Allows the initialization of e.g. model values
     */
    init() {};
    
    /**
     * Called each time, the component is connected to the DOM. Here is a good place to register some listeners.
     */
    attached() {};
    
    /**
     * Called each time, the component is removed from the DOM. Event listeners should be removed here. 
     */
    detached() {};
	
	/**
	 * Registers a callback that is called when the given property changes the dataBindingModel. The property may contain dot to access inner values.
	 * @param {string} propertyName the name of the property to by watched
	 * @param {function} callback the function that is called in case of changes. will provide the new value as single argument
	 * @return a function that can be called to deregister the watcher
	 */
	watch(propertyName, callback) {
		if (!this.#watchers.has(propertyName)) {
			this.#watchers.set(propertyName, []);
		}
		this.#watchers.get(propertyName).push(callback);
		return () => {
			let index = this.#watchers.get(propertyName).indexOf(callback);
			if (index >= 0) {
				this.#watchers.get(propertyName).splice(index,1);
				if (this.#watchers.get(propertyName).length==0) {
					this.#watchers.delete(propertyName);
				}
			}
		};
	}
        
    /*****************************************************************************************************************************************************************************************************************************/
    /*****************************************************************************************************************************************************************************************************************************/
    /** API END                                                                                                                                                                                                                 **/
    /*****************************************************************************************************************************************************************************************************************************/
    /*****************************************************************************************************************************************************************************************************************************/
    
    // private members
    #boundProperties = {};
    #initialized = false;
	#watchers = new Map();
    
    /**
     * Recursivly proxies the dataBindingModel to get change events when a value is changed
     * @param dataBindingModel the model to create a proxy with
     * @param propertyName the name of the property in the given model
     * @param parentProperty the parent property path to allow references to e.g. innerModel.child.leaf with dor syntax
     */
    #proxyDataBindingModel(dataBindingModel = this.dataBindingModel, propertyName = undefined, parentProperty = '') {
        let instance = this;
        let dataBindingModelToProxy = dataBindingModel;
        
        if (propertyName != undefined) {
            dataBindingModelToProxy = dataBindingModel[propertyName];
        }
        
        let proxy = new Proxy(dataBindingModelToProxy,{
            set(obj, prop, value) {
                let returnValue = Reflect.set(...arguments);
                
                if(typeof obj[prop] == 'object') {
                    instance.#proxyDataBindingModel(obj, prop, parentProperty.length==0?prop:(parentProperty+'.'+prop));
                }
                
                if (parentProperty.length>0) {
                    prop = parentProperty+'.'+prop;
                }
                propertyName = prop;
                prop = 'dataBindingModel.'+prop;
                instance.#updateBoundProperties(prop);
				
				if (instance.#watchers.has(propertyName)) {
					let watcher = instance.#watchers.get(propertyName);
					for (let wtch of watcher) {
						wtch(obj[propertyName]);
					}
				}
				
                return returnValue;
            }
        });
        
        let keys = Object.keys(dataBindingModelToProxy);
        for (let key of keys) {
            let val = dataBindingModelToProxy[key];
            if (typeof val == 'object') {
                this.#proxyDataBindingModel(dataBindingModelToProxy, key, parentProperty.length==0?key:(parentProperty+'.'+key));
            }
        }
        
        if (propertyName == undefined) {
            this.dataBindingModel = proxy;
        } else {
            dataBindingModel[propertyName] = proxy;            
        }
    };
    
    /**
     * Updates all registered nodes for the given property name. It will change the textcontent or the value, according to the element type.
     * @param propName the name of the property that should be updates in the DOM
     */
    #updateBoundProperty(propName) {
        let nodeList = this.#boundProperties[propName];
        for (let node of nodeList) {
            if (node.nodeType == Node.ELEMENT_NODE) {
                let val = Kytos.getModelValue(propName, this);
                if ('value' in node) {
                    if (node.value != val) {
                        node.value = val;
                    }
                } else {
                    if (node.textContent != val) {
                        node.textContent = val;
                    }
                }
            } else if (node.nodeType == Node.TEXT_NODE) {
                let content = node.parentNode.kOriginalContent;
                content = content.replace('{{'+propName+'}}', Kytos.getModelValue(propName, this));
                
                let startIndex = content.indexOf('{{');
                while (startIndex >= 0) {
                    let endIndex = content.indexOf('}}', startIndex);
                    if (endIndex >= 0) {
                        let property = content.substring(startIndex+2, endIndex);
                        content = content.replace('{{'+property+'}}', Kytos.getModelValue(property, this));
                    }
                    startIndex = content.indexOf('{{', endIndex);
                }
                if (node.data != content) {
                    node.data = content;
                }
            }
        }
    };
    
    /**
     * Updates all registered bound property nodes. Optionally takes a filter to only update a subset of properties
     * @param startsWithFilter a string with a property name of values that should be updated. The filter 'test' will update the property test and all properties that start with 'test.' like 'test.subValue'.  
     */
    #updateBoundProperties(startsWithFilter = undefined) {
        let propNames = Object.keys(this.#boundProperties);
        if (propNames.length>0) {
            for (let propName of propNames) {
                if (startsWithFilter != undefined) {
                    if (!(propName == startsWithFilter || propName.startsWith(startsWithFilter + '.'))) {
                        continue;
                    }
                }
                this.#updateBoundProperty(propName);
            }
        }
    };
    
    /**
     * Iterates all node of the current component to find all bindings with mustach syntax like '{{myValue}}' or kytos bindings like 'k-bind="myValue"'
     */
    #collectBindings() {
		this._kytosAttributeNodes = new Map();
		let instance = this;
        // find all text nodes with containing {{...}} bindings
        const treeWalker = document.createTreeWalker(this, NodeFilter.SHOW_TEXT | NodeFilter.SHOW_ELEMENT, (node) => {
			// prevent the collection of binding in inner kytos elements
			if (node.parentNode != undefined && node.parentNode.nodeType == Node.ELEMENT_NODE && node.parentNode instanceof KytosElement && node.parentNode != instance) {
				return NodeFilter.FILTER_REJECT;
			}
			if (node.parentNode != undefined && node.parentNode.kytosTemplateNode) {
				return NodeFilter.FILTER_REJECT;
			}
			return NodeFilter.FILTER_ACCEPT;
		});
        while ( treeWalker.nextNode() ) {
            const node = treeWalker.currentNode;
            
            if (node.nodeType == Node.ELEMENT_NODE) {
                let property = node.getAttribute('k-bind');
                if (property != undefined) {
                    let nodeList = this.#boundProperties[property] || [];
                    this.#boundProperties[property] = nodeList;
                    nodeList.push(node);
                }
                
                for (let attrName of node.getAttributeNames()) {
                    if (KytosAttribute._hasAttributeDefined(attrName)) {
						if (!this._kytosAttributeNodes.has(attrName)) {
							this._kytosAttributeNodes.set(attrName, []);
						}
						if (KytosAttribute._skipsInnerElements(attrName)) {
							this._kytosAttributeNodes.get(attrName).push(node);
							node.kytosTemplateNode = true;
							break;
						}
						this._kytosAttributeNodes.get(attrName).push(node);
                    }
                }
                
            } else if (node.nodeType == Node.TEXT_NODE) {
                let startIndex = node.data.indexOf('{{');
                while (startIndex >= 0) {
                    let endIndex = node.data.indexOf('}}', startIndex);
                    if (endIndex >= 0) {
                        let property = node.data.substring(startIndex+2, endIndex);
                        node.parentNode.kOriginalContent = node.data.toString();
                        
                        let nodeList = this.#boundProperties[property] || [];
                        this.#boundProperties[property] = nodeList;
                        nodeList.push(node);
                    }
                    startIndex = node.data.indexOf('{{', endIndex);
                }
            }
        }
		
		for (let [attrName, nodes] of this._kytosAttributeNodes) {
			for (let node of nodes) {
				KytosAttribute._bindAttribute(attrName, this, node);
			}
		}
    };
    
    /**
     * Called when the component is connected to the DOM. Will initialize the component and add the optoinal template to the component.
     * Additionally angularJS binding is prevented by adding the corresponding attribute.
     */
    connectedCallback() {
        if (!this.#initialized) {
            this.#initialized = true;
            this.init();
            this.#proxyDataBindingModel();
        }
        
        this.setAttribute('ng-non-bindable', '');
        
        if (this.template != undefined) {
            this.innerHTML = this.template;
            this.#collectBindings();
            this.#updateBoundProperties();
        }
        
        this.attached();
    };
    
    /**
     * Called when the component is removed from the DOM. Will clear all bindings.
     */
    disconnectedCallback() {
        let props = Object.getOwnPropertyNames(this.#boundProperties);
        for (let i = 0; i < props.length; i++) {
          delete this.#boundProperties[props[i]];
        }
		
        // Remove all collected attribute bindings
        if (this._kytosAttributeNodes != undefined) {
            for (let [attrName, nodes] of this._kytosAttributeNodes) {
                for (let node of nodes) {
                    KytosAttribute._unbindAttribute(attrName, this, node);
                }
            }
        }
		
        this.detached();
    };
    
}
