import { KytosAttribute } from './kytos-attribute.js';
import { EventEmitter } from './eventemitter.js';

/**
 * KytosDropdown is a custom dropdown component that extends KytosAttribute,
 * providing open/close logic, keyboard navigation, and event handling for dropdown menus.
 *
 * This class manages dropdown state, accessibility, and event propagation,
 * including support for legacy and custom events, keyboard navigation (Arrow keys, Escape),
 * and click handling both inside and outside the dropdown. 
 * 
 * Usage: attribute "dropdown-toggle" on the source element eg. button
 * 
 * <div class="dropdown" k-dropdown>                                   												// optional class "dropup"
 *     <button type="button" class="btn" dropdown-toggle></button>     												// optional class "btn-withcaret"
 *     <ul class="dropdown-menu">                                      												// optional class "dropdown-menu-right"
 *         <li class="dropdown-header"></li>                           												// optional class "dropdown-header"  a click will not collapse the dropdown
 *         <li class="stay-open"><a href=""><span>List items that stays open on click</span></a></li>               // optional class "stay-open" a click will not collapse the dropdown
 *         <li class="divider"></li>                                   												// optional class "divider"  a click will not collapse the dropdown
 *         <li k-dropdown>																							// Starts a new submenu from here	
 *  			<a href="" dropdown-toggle><span>Sub menu here</span></a>  											// toggle for a submenu
 *  			<ul class="dropdown-menu">																			// submenu that opens on click
 *  				<li><a href="">Sub menu item 1</a></li>
 *  				<li><a href="">Sub menu item 2</a></li>
 *  			</ul>
 * 			</li>                                   
 *     </ul>
 * </div>
 * 
 * Events:
 *  - "hide.dropdown"  : Triggered before a dropdown starts to close.
 *  - "hidden.dropdown": Triggered after a dropdown has closed.
 *  - "show.dropdown"  : Triggered before a dropdown starts to open.
 *  - "shown.dropdown" : Triggered after a dropdown has opened.
 */

export class KytosDropdown extends KytosAttribute {

	/**
	 * Opens the dropdown menu for the given node.
	 * Fires a global event to hide other dropdowns, dispatches a custom event to show this dropdown,
	 * and focuses the first enabled menu item.
	 *
	 * @private
	 * @param {HTMLElement} node - The DOM node representing the dropdown container.
	 */
	#open(node) {
		EventEmitter.fire('hide.angular.dropdown');
		node.dispatchEvent(new CustomEvent('show.dropdown'));
		let menu = node.querySelector('.dropdown-menu');
		menu.querySelector('li:not(.disabled):not([disabled]) > a').focus();
	};

	/**
	 * Dispatches a custom 'hide.dropdown' event on the specified DOM node to signal that the dropdown should be hidden.
	 *
	 * @private
	 * @param {Node} node - The DOM node on which to dispatch the 'hide.dropdown' event.
	 */
	#close(node) {
		node.dispatchEvent(new CustomEvent('hide.dropdown'));
	};


	/**
	 * Attaches dropdown behavior to a given node, enabling open/close logic,
	 * keyboard navigation, and event handling for a custom dropdown component.
	 *
	 * @param {HTMLElement} kytosElement - The custom element instance this behavior is attached to.
	 * @param {HTMLElement} node - The DOM node representing the dropdown container.
	 * @param {string} attrName - The attribute name associated with this dropdown instance.
	 *
	 * @throws {Error} If no toggle element is found within the node.
	 *
	 * @listens show.dropdown - Opens the dropdown and sets up event listeners.
	 * @listens hide.dropdown - Closes the dropdown and removes event listeners.
	 * @fires shown.dropdown - Dispatched when the dropdown is shown.
	 * @fires hidden.dropdown - Dispatched when the dropdown is hidden.
	 */
	attached(kytosElement, node, attrName) {
		let toggle = node.querySelector('[dropdown-toggle]');
		if (toggle == undefined) {
			throw new Error('No toggle element found for dropdown');
		}

		/**
		 * Handles the legacy event to hide the dropdown.
		 * Dispatches a custom 'hide.dropdown' event from the specified node.
		 *
		 * @param {Event} event - The event object triggering the handler.
		 */
		const onLegacyHideDropdown = (event) => {
			node.dispatchEvent(new CustomEvent('hide.dropdown'));
		};

		/**
		 * Handles document click events to manage the open/close state of a dropdown component.
		 * Closes the dropdown if the click is outside the dropdown node.
		 * If the click is inside the dropdown:
		 *   - Prevents closing if the clicked list item has the 'stay-open' class or 'k-dropdown' attribute.
		 *   - Otherwise, closes the dropdown.
		 *
		 * @param {MouseEvent} event - The click event triggered on the document.
		 */
		const onDocumentClick = (event) => {
			let target = event.target;

			if (!node.contains(target)) {
				this.#close(node);
				return;
			}

			// We clicked inside our dopdown
			let listItem = target.closest('li');
			if (listItem != undefined) {
				event.preventDefault();
				if (listItem.classList.contains('stay-open')) {
					return;
				}
				if (listItem.hasAttribute('k-dropdown')) {
					return;
				};
				this.#close(node);
				node.querySelector('[dropdown-toggle]').focus();
			}
		};

		/**
		 * Handles keyboard navigation for dropdown menu items.
		 * - ArrowDown: Moves focus to the next enabled menu item.
		 * - ArrowUp: Moves focus to the previous enabled menu item.
		 * - Escape: Closes the dropdown menu and returns focus to the toggle element.
		 *
		 * @param {KeyboardEvent} event - The keyboard event triggered by user interaction.
		 */
		const onMenuKeyDown = (event) => {
			if (event.key === 'ArrowDown') {
				event.preventDefault();
				event.stopPropagation();
				let items = [...node.querySelectorAll('.dropdown-menu > li:not(.disabled):not([disabled]) a')];
				let currentLi = event.target.closest('li a');
				let index = items.indexOf(currentLi) + 1;
				if (index < items.length) {
					items[index].focus();
				}
			} else if (event.key === 'ArrowUp') {
				event.preventDefault();
				event.stopPropagation();
				let items = [...node.querySelectorAll('.dropdown-menu > li:not(.disabled):not([disabled]) a')];
				let currentLi = event.target.closest('li a');
				let index = items.indexOf(currentLi) - 1;
				if (index >= 0) {
					items[index].focus();
				}
			} else if (event.key === 'Escape') {
				event.preventDefault();
				event.stopPropagation();
				this.#close(node);
				toggle.focus();
			}
		};

		/**
		 * Handles actions to perform when the dropdown is opened.
		 * - Adds a keydown event listener to the dropdown node for menu navigation.
		 * - Adds a click event listener to the document to handle outside clicks.
		 * - Dispatches a custom 'shown.dropdown' event from the dropdown node.
		 */
		const onDropDownOpened = () => {
			node.addEventListener('keydown', onMenuKeyDown);
			node.ownerDocument.addEventListener('click', onDocumentClick, true);
			node.dispatchEvent(new CustomEvent('shown.dropdown'));
		};

		/**
		 * Handles cleanup when the dropdown menu is closed.
		 * Removes event listeners for 'keydown' and 'click' events,
		 * and dispatches a custom 'hidden.dropdown' event on the node.
		 */
		const onDropDownClosed = () => {
			node.removeEventListener('keydown', onMenuKeyDown);
			node.ownerDocument.removeEventListener('click', onDocumentClick);
			node.dispatchEvent(new CustomEvent('hidden.dropdown'));
		};

		node.addEventListener('hide.dropdown', (event) => {
			if (!event.defaultPrevented) {
				EventEmitter.off('hide.kytos.dropdown', onLegacyHideDropdown);
				node.classList.remove('open');
				onDropDownClosed();
			}
		});

		node.addEventListener('show.dropdown', (event) => {
			if (!event.defaultPrevented) {
				EventEmitter.on('hide.kytos.dropdown', onLegacyHideDropdown);
				node.classList.add('open');
				onDropDownOpened();
			}
		});

		toggle.addEventListener('click', (event) => {
			if (node.classList.contains('open')) {
				this.#close(node);
			} else {
				let toggle = node.querySelector('[dropdown-toggle]')
				if (toggle.hasAttribute('disabled') || toggle.classList.contains('disabled')) {
					return;
				}
				this.#open(node);
			}
		});

		/**
		 * Handles keyboard navigation for the dropdown toggle element.
		 * 
		 * - For non-anchor toggles, DOWN and UP arrow keys will open or close the dropdown.
		 * - For anchor toggles, LEFT and RIGHT arrow keys will open or close the dropdown.
		 * - Prevents default browser behavior and stops event propagation when handling keys.
		 *
		 * @param {KeyboardEvent} event - The keyboard event triggered on keydown.
		 */
		const onToggleKeyDown = (event) => {
			if (event.key === 'ArrowDown' || event.key === 'ArrowUp') {
				if (toggle.tagName === 'A') {
					return;
				}
				event.preventDefault();
				event.stopPropagation();
				if (node.classList.contains('open')) {
					this.#close(node);
				} else {
					this.#open(node);
				}
			}
			if (event.key === 'ArrowLeft' || event.key === 'ArrowRight') {
				if (toggle.tagName !== 'A') {
					return;
				}
				event.preventDefault();
				event.stopPropagation();
				if (node.classList.contains('open')) {
					this.#close(node);
				} else {
					this.#open(node);
				}
			}
		};

		toggle.addEventListener('focus', (event) => {
			toggle.addEventListener('keydown', onToggleKeyDown);
		});

		toggle.addEventListener('blur', (event) => {
			toggle.removeEventListener('keydown', onToggleKeyDown);
		});

	};

};

KytosAttribute.define('k-dropdown', KytosDropdown, false);
