/**
 * ErrorService is a utility class for managing error messages in the Kytos GUI.
 * It provides methods to show and clear error messages in a specified container.
 */
export class ErrorService {

	/*****************************************************************************************************************************************************************************************************************************/
	/*****************************************************************************************************************************************************************************************************************************/
	/** API methods and members                                                                                                                                                                                                 **/
	/*****************************************************************************************************************************************************************************************************************************/
	/*****************************************************************************************************************************************************************************************************************************/

	/**
	 * Shows an error message in the specified container.
	 * @param {HTMLElement|string} errorContainer - The container where the error message will be displayed. Can be a selector string or an HTMLElement.
	 * @param {string} message - The error message to display.
	 * @param {string} [id=''] - An optional identifier for the error message, used to clear it later.
	 */
	static showError(errorContainer, message, id = '') {

		if (typeof errorContainer === 'string') {
			errorContainer = document.querySelector(errorContainer);
		}
		if (errorContainer instanceof HTMLElement) {

			let closable = errorContainer.getAttribute('data-closable') != 'false';

			let errorTemplateStart = `
				<div class="alert-box alert-box-danger" error-id="${id}">
					<i class="alert-box-icon icon-notification-error"></i>
					<div class="alert-box-text">${message}</div>
			`;
			let errorTemplateEnd = `
				</div>
			`;
			let closeButton = `
				<div class="alert-box-close">
                	<button type="button" class="btn btn-link"><i class="icon-close-notification"></i></button>
                </div>
			`;
			let errorTemplate = errorTemplateStart;
			if (closable) {
				errorTemplate += closeButton;
			}
			errorTemplate += errorTemplateEnd;

			errorContainer.insertAdjacentHTML('beforeend', errorTemplate);

			if (closable) {
				let closeButtonElement = errorContainer.querySelector('.alert-box-close button');
				closeButtonElement.addEventListener('click', (event) => {
					event.currentTarget.closest('.alert-box').remove();
				});
			}
		} else {
			console.error('ErrorService.showError: Invalid error container provided.');
			console.error(message);
		}
	};

	/**
	 * Clears an error message from the specified container.
	 * @param {HTMLElement|string} errorContainer - The container from which the error message will be removed. Can be a selector string or an HTMLElement.
	 * @param {string} [id=''] - The identifier of the error message to clear. If not provided, all error messages will be cleared.
	 */
	static clearError(errorContainer, id = '') {
		if (typeof errorContainer === 'string') {
			errorContainer = document.querySelector(errorContainer);
		}
		if (errorContainer instanceof HTMLElement) {
			let errorElements = errorContainer.querySelectorAll(`[error-id="${id}"]`);
			if (errorElements.length > 0) {
				for (let errorElement of errorElements) {
					errorElement.remove();
				}
			}
		} else {
			console.error('ErrorService.clearError: Invalid error container provided.');
		}
	};

	/*****************************************************************************************************************************************************************************************************************************/
	/*****************************************************************************************************************************************************************************************************************************/
	/** API END                                                                                                                                                                                                                 **/
	/*****************************************************************************************************************************************************************************************************************************/
	/*****************************************************************************************************************************************************************************************************************************/

};
